#!/usr/bin/env sh
# SPDX-License-Identifier: AGPL-3.0-or-later
# SPDX-FileCopyrightText: 2024-2025 JayVii <jayvii[AT]posteo[DOT]de>

# Configuration ----------------------------------------------------------------

# ensure pretty names are always in english
LANG="en"

# Fetch Inputs -----------------------------------------------------------------

# all given arguments are interpreted as git-directories
DIRS="$@"

# Output Functions -------------------------------------------------------------

# HTML output
html_output() {
  # grab function input
  DTE="$1"
  CMT="$2"
  OPC="$3"
  TTL=""
  # if date is present, add it to the title
  if [ ! -z "$DTE" ]; then
    TTL=${TTL}$(date --date "${DTE}-01" "+%b, %Y");
  fi
  # if date and comment are present, add separator to the title
  if [ ! -z "$DTE" ] && [ ! -z "$CMT" ]; then
    TTL="${TTL}: "
  fi
  # if comment is present, add it to the title
  if [ ! -z "$CMT" ]; then TTL="${TTL}${CMT} commits"; fi
  # re-scale opacity linearly between 10-100
  OPC=$(echo "(($OPC * 90) / 100) + 10" | bc)
  # create HTML arguments
  ID="activity-$DTE"
  CL="activitypoint"
  STL="opacity:calc($OPC/100);"
  ARGS="id=\"$ID\" class=\"activitypoint\" style=\"$STL\" title=\"$TTL\""
  # return HTML string
  printf "<span $ARGS>&#9632;</span>"
}

# Collect activity -------------------------------------------------------------

# By Months
ACTIVITY=""
for DIR in $DIRS; do
  ACTIVITY="${ACTIVITY}$(
    git \
      -C $DIR \
      log --date=short --pretty=format:%ad | \
      sed -E -e 's/^([0-9]{4}-[0-9]{2})-.*$/\1/g'
  )\n"
done

# aggregate activity
AGGREGATE=$(
  printf "$ACTIVITY" | \
    sort | \
    uniq -c
)

# Process Activity -------------------------------------------------------------

# get timeframe
MIN_TIME=$(echo "$AGGREGATE" | head -n 1 | awk '{ print $NF }')
MAX_TIME=$(date +%Y-%m)

# get value range
MIN_VAL=0
MAX_VAL=$(echo "$AGGREGATE" | awk '{ print $1 }' | sort -n | tail -n 1)
if [ -z "$MAX_VAL" ]; then MAX_VAL=1; fi

# Generate output --------------------------------------------------------------

# initilise html
MIN_TIME_PRETTY=$(date --date "${MIN_TIME}-01" "+%b, %Y")
MAX_TIME_PRETTY=$(date --date "${MAX_TIME}-01" "+%b, %Y")
HTML="<h6>Activity: ${MIN_TIME_PRETTY} - ${MAX_TIME_PRETTY}</h6>"

# cycle through months from earliest timepoint
CUR_TIME=$MIN_TIME
while [ "$CUR_TIME" != "$MAX_TIME" ]; do # FIXME: break occurs here

  # fetch current value
  CUR_VAL=$(echo "$AGGREGATE" | grep $CUR_TIME | awk '{ print $1 }')
  if [ -z "$CUR_VAL" ]; then CUR_VAL=0; fi

  # normalise current value [0, 100]
  NRM_VAL=$(echo "${CUR_VAL} * 100 / ${MAX_VAL}" | bc)
  if [ -z "$NRM_VAL" ]; then NRM_VAL=0; fi

  # output HTML
  HTML=${HTML}$(html_output "$CUR_TIME" "$CUR_VAL" "$NRM_VAL")

  # update current time
  CUR_TIME=$(date --date "${CUR_TIME}-01 +1 month" +%Y-%m)

  # reset variables for next round
  unset CUR_VAL
  unset NRM_VAL

done

# add legend
HTML="${HTML}<br><p id=\"activitylegend\">Less "
for LV in $(seq 0 25 100); do

  # Calculate label value
  LND_VAL=$(($LV * $MAX_VAL / 100))

  # Generate HTML
  HTML=${HTML}$(html_output "" $LND_VAL $LV)

done
HTML="${HTML} More</p>"

# add "activitybar" div
HTML="<div id=\"activitybar\">$HTML</div>"

# Print output -----------------------------------------------------------------

printf "$HTML"
