#!/usr/bin/env bash

# exit on error
set -eo pipefail

# configuration
config_dir="$HOME/.config/helix"
config="$config_dir/config.toml"
config_theme="$config_dir/autothemes.toml"

# gather dark and light themes from config file
if [ -f "$config_theme" ]; then
  dark_theme=$(
    grep -e "^\s*theme_dark" "$config_theme" | \
      sed -E -e 's/^\s*theme_dark\s*=\s*\"([^"]+)\".*$/\1/'
  )
  light_theme=$(
    grep -e "^\s*theme_light" "$config_theme" | \
      sed -E -e 's/^\s*theme_light\s*=\s*\"([^"]+)\".*$/\1/'
  )
else
  echo "
    Config file $config_theme does not exist. Please create the file, e.g.:
    
    theme_dark = \"adwaita-dark\"
    theme_light = \"adwaita-light\"
  "
  exit 1
fi

# functions
function get_theme {
  set +e
  grep -c -e "^\s*theme\s*=\s*\"${1}\"" ${config}
  set -e
}
function set_theme {
  sed -E -e "s/^\s*(theme\s*=\s*)\"[^\"]+\"/\1\"${1}\"/" -i ${config}
}

# check whether helix currently uses the dark theme
is_dark=$(get_theme "${dark_theme}")
is_light=$(get_theme "${light_theme}")

# check whether input was given
if [ -z $1 ]; then
  if [[ $is_dark != 0 ]]; then
    target="light"
  else
    target="dark"
  fi
else
  target="$1"
fi


if [ "$target" == "dark" ]; then
  if [[ $is_dark != 1 ]]; then
    set_theme  "${dark_theme}"
    echo "[✓] Switched helix to dark!"
  else
    echo "[-] Did not switch helix theme."
  fi
else
  if [[ $is_light != 1 ]]; then
    set_theme  "${light_theme}"
    echo "[✓] Switched helix to light!"
  else
    echo "[-] Did not switch helix theme."
  fi
fi
